/*
   license:

    ------------------------ Apache Version 2.0 license -------------------------
       Copyright (C) 2024 Jan Voglmüller 

       Licensed under the Apache License, Version 2.0 (the "License");
       you may not use this file except in compliance with the License.
       You may obtain a copy of the License at

           http://www.apache.org/licenses/LICENSE-2.0

       Unless required by applicable law or agreed to in writing, software
       distributed under the License is distributed on an "AS IS" BASIS,
       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
       See the License for the specific language governing permissions and
       limitations under the License.
    -----------------------------------------------------------------------------
*/

/** This extension for ooRexx and BSF4ooRexx adds classes to facilitate the creation of business charts with jdor.
*  @author Jan Voglmüller
*  @version 1.0
*/

/** This class represents two-dimensional data points.
*/
::class data
    ::attribute xValue
    ::attribute yValue

    /** Constructor, fetches xValue and yValue.
    *  @param xValue the x value
    *  @param yValue the y value
    */
    ::method init
        expose xValue yValue
        use arg xValue, yValue
        if dataType(yValue, "N") <> .true then
            do
                say "ERROR -->" pp(yValue)
                say "yValues have to be a number!"
                yValue = -100
            end
        if yValue < 0 then
            do
                say "yValues should be positive, I will correct that" pp(yValue) "for you :)"
                yValue = abs(yValue)
            end

    /** Returns xValue and yValue.
    *  @return xValue and yValue.
    */
    ::method allValues
        expose xValue yValue
        return xValue yValue

/** This class manages a collection of data points.
*/
::class dataSet PUBLIC
    ::attribute values

    /** Constructor, initializes an empty array of values.
    */
    ::method init
        expose values
        values = .array~new

    /** Adds a new data point to the collection.
    *  @param x the xValue
    *  @param y the yValue
    */
    ::method addValue
        expose values
        use arg x, y
        values~append(.data~new(x, y))

    /** Prints all data points.
    */
    ::method allValues
            expose values
            do counter index data over values
                say "#"index "Values:" data~allValues
            end

    /** Finds the maximum value of either x or y from the data points.
    *  @param xy specifies whether to find the maximum of x or y
    *  @return the maximum value
    */
    ::method maxValue
        expose values
        use arg xy
        maxValue = 0
        do data over values
            if xy = "x" then
               maxValue = max(maxValue, data~xValue)
            else
               maxValue = max(maxValue, data~yValue)
        end
        return maxValue

    /** Sums up all y values in the collection.
    *  @return the sum of y values
    */
    ::method sum
        expose values
        sum = 0
        do data over values
            sum += data~yValue
        end
        return sum

    /** Gets the number of data points.
    *  @return the number of data points
    */
    ::method size
        expose values
        return values~size

    /** Removes a data point at a specific index.
    *  @param del the index of the data point to be removed
    */
    ::method remove
        expose values
        use arg del
        newArray = .array~new
        do counter index data over values
            if index <> del then
                newArray~append(data)
        end
        values = newArray

    /** Clears all data points.
    */
    ::method clear
        expose values
        values~empty

/** This class creates different types of charts.
*/
::class JChart PUBLIC
    ::attribute version class

    /** Constructor, initializes chart creator with default dimensions and setting the version.
    *  @param minWidth the minimum width of the window (default: 800)
    *  @param minHeight the minimum height of the window (default: 500)
    */
    ::method activate class      -- runs once after all classes got created, needed to access .JBCconstants class
       expose minWidth minHeight version
       use arg minWidth = 800, minHeight = 500
       version = .JBCconstants~version

    /** Creates a chart based on the specified parameters.
    *  @param chartType the type of chart
    *  @param dataSet the dataset for the chart
    *  @param title (optional) the title of the chart (default: "Untitled")
    *  @param xLabel (optional) the label for the x-axis (default: "x-Axis")
    *  @param yLabel (optional) the label for the y-axis (default: "y-Axis")
    *  @param width (optional) the width of the chart (default: 800)
    *  @param height (optional) the height of the chart (default: 500)
    *  @return the created chart object
    */
    ::method createChart class
        expose minWidth minHeight
        use arg chartType, dataSet, title = "Untitled", xLabel = "x-Axis", yLabel = "y-Axis", width = 800, height = 500
        /* set min width/height */
        width = max(width,  minWidth)
        height = max(height, minHeight)

        /* create selected chart */
        clzChart=.context~package~classes~entry(chartType)
        select case chartType
            when "ColumnChart", "BarChart", "LineChart", "PointChart" then
                chart = clzChart~new(dataSet, title, xLabel, yLabel, width, height)
            when "PieChart", "RingChart" then
                chart = clzChart~new(dataSet, title, width, height)
            otherwise
                say "Unknown chart type:" chartType
                chart= .nil
        end
        return chart

    /** Creates a multi chart or mixed chart based on the provided charts.
    *  @param chart1 the first chart
    *  @param chart2 the second chart
    *  @param chart3 (optional) the third chart
    *  @param chart4 (optional) the fourth chart
    *  @param chart5 (optional) the fifth chart
    *  @return the combined chart
    */
    ::method combineCharts class
        use arg chart1, chart2, chart3 = .nil, chart4 = .nil, chart5 = .nil
        if chart1~class~id = chart2~class~id then
            select case chart1~class~id
                when "MULTICHART" then
                    goto mixedCharts
                otherwise
                    do
                        chart = .multiChart~new(chart1, chart2, chart3, chart4, chart5)
                        return chart
                    end
            end
        else
            goto mixedCharts

        mixedCharts:
        chart = .mixedChart~new(chart1, chart2)
        return chart

    /** Creates a stacked chart.
    *  @param chart1 the first chart
    *  @param chart2 the second chart
    *  @param chart3 (optional) the third chart
    *  @param chart4 (optional) the fourth chart
    *  @param chart5 (optional) the fifth chart
    *  @return the stacked chart
    */
    ::method stackCharts class
        use arg chart1, chart2, chart3 = .nil, chart4 = .nil, chart5 = .nil
        chart = .stackChart~new(chart1, chart2, chart3, chart4, chart5)
        return chart

/** This class represents a drawing canvas and creates a window to display a chart.
*/
::class canvas
    ::attribute handler

    /** Constructor, initializes the canvas with specified parameters.
    *  @param title the title of the canvas
    *  @param width the width of the canvas
    *  @param height the height of the canvas
    *  @param background the background color of the canvas
    */
    ::method init
        expose handler
        use arg title, width, height, background
        /* start handler */
        handler = .bsf~new("org.oorexx.handlers.jdor.JavaDrawingHandler")
        call BsfCommandHandler "add", "JDOR", handler
        /* create window */
        address jdor
        newImage width height
            /* add background */
        color background
        fillRect width height
            /* add image */
        --fileName = "test.png" --images[c]
        --imageFromFile="img_"c
        --loadImage imageFromFile fileName
        --drawImage imageFromFile
            /* finish window */
        winTitle title
        winShow
        reset

/** This class represents a coordinate system for chart drawing.
*/
::class coorSys
    ::attribute dataSet
    ::attribute unitNames

    /** Constructor, initializes coordinate system with chart parameters.
    *  @param chartType the type of chart
    *  @param dataSet the dataset for the chart
    *  @param title the title of the chart
    *  @param xLabel the label for the x-axis
    *  @param yLabel the label for the y-axis
    *  @param width the width of the window
    *  @param height the height of the window
    *  @param unitNames an array with the names for big numbers (>999)
    */
    ::method init
        expose chartType dataSet title xLabel yLabel width height unitNames chartWidth chartHeight yLineNum yLineDist charWidth
        use arg chartType, dataSet, title, xLabel, yLabel, width, height, unitNames
        /* setup parameters */
        chartWidth = sizeXToChart(width)
        chartHeight = sizeYToChart(height)
        yLineNum = 10
        yLineDist = chartHeight / yLineNum
        charWidth = 5

    /** Adds a title to the chart.
    *  @return 1 on success
    */
    ::method addTitle
        expose width title charWidth
        address jdor
        call resetFormat
        moveTo (width / 2 - title~length * 6.5) (.JBCconstants~spacerEdge / 1.5)
        fontstyle 1
        fontsize 25
        font arial "Arial"
        drawString title
        call resetFormat
        return 1

    /** Draws the x-axis with optional percent formatting.
    *  @param xy the axis type ("x" or "y")
    *  @param percent whether to format values as percentages
    */
    ::method drawXAxis
        expose yLineNum chartWidth chartHeight dataSet
        use arg xy = "x", percent = .false
        yLineDist = chartWidth / yLineNum
        address jdor
        do i = yLineNum to 0 by -1
            /* lines */
            moveTo (.JBCconstants~spacerEdge + yLineDist * i) .JBCconstants~spacerEdge
            color lightgrey
            drawLine (.JBCconstants~spacerEdge + yLineDist * i) (chartHeight + .JBCconstants~spacerEdge)
            /* values */
            xLineValue = dataSet~maxValue(xy) / yLineNum * i
            moveTo (.JBCconstants~spacerEdge + chartWidth / yLineNum * i - xLineValue~length * .JBCconstants~charLength) (.JBCconstants~spacerEdge + chartHeight + 17)
            /* check for needed format changes */
            if i = yLineNum then
                fc = self~checkValue(xLineValue)
            /* change format if needed */
            xLineValue = self~formatValue(xLineValue, fc)
            color black
            drawString xLineValue
        end
        if percent then
            fc = "in %"
        moveTo (.JBCconstants~spacerEdge + chartWidth) (.JBCconstants~spacerEdge + chartHeight + 32)
        drawString fc
        call resetFormat

    /** Draws the y-axis with optional percent formatting.
    *  @param percent whether to format values as percentages
    */
    ::method drawYAxis
        expose yLineNum yLineDist chartWidth dataSet fc fcn
        use arg percent = .false
        address jdor
        do i = 0 to yLineNum
            /* lines */
            moveTo .JBCconstants~spacerEdge (.JBCconstants~spacerEdge + yLineDist * i)
            color lightgrey
            drawLine (chartWidth + .JBCconstants~spacerEdge) (.JBCconstants~spacerEdge + yLineDist * i)
            /* values */
            yLineValue = dataSet~maxValue("y") / yLineNum * (yLineNum - i)
            color black
            /* check for needed format changes */
            if i = 0 then
                fc = self~checkValue(yLineValue)
            /* change format if needed */
            yLineValue = self~formatValue(yLineValue, fc)
            moveTo (.JBCconstants~spacerEdge - 20 - yLineValue~length * 4.5) (.JBCconstants~spacerEdge + yLineDist * i + 4.5)
            drawString yLineValue
        end
        if percent then
            fc = "in %"
        moveTo (.JBCconstants~spacerEdge / 2 - fc~length * 4) (.JBCconstants~spacerEdge / 1.2)
        drawString fc
        call resetFormat

    /** Draws a secondary y-axis on the right side.
    *  @param axisLabels? whether to display axis labels
    */
    ::method drawYAxis2
        expose yLineNum chartWidth yLineDist dataSet chartHeight height yLabel charWidth
        use arg axisLabels? = .true
        address jdor
        /* yValues */
        do i = 0 to yLineNum
            moveTo (.JBCconstants~spacerEdge + chartWidth + 10) (.JBCconstants~spacerEdge + yLineDist * i + 4.5)
            yLineValue = dataSet~maxValue("y") / yLineNum * (yLineNum - i)
            /* check for needed format changes */
            if i = 0 then
                fc = self~checkValue(yLineValue)
            /* change format if needed */
            yLineValue = self~formatValue(yLineValue, fc)
            drawString yLineValue
        end
        moveTo (.JBCconstants~spacerEdge + chartWidth - 10) (.JBCconstants~spacerEdge / 1.2)
        drawString fc
        /* axis */
        moveTo (.JBCconstants~spacerEdge + chartWidth) .JBCconstants~spacerEdge
        stroke strokeThick3 3
        drawLine (.JBCconstants~spacerEdge + chartWidth) (.JBCconstants~spacerEdge + chartHeight)
        call resetFormat
        /* label */
        if axisLabels? then
            do
                translate (.JBCconstants~spacerEdge + chartWidth + .JBCconstants~spacerEdge / 1.1) (height / 2 + yLabel~length * charWidth)
                moveto 0 0
                rotate 270
            end
        fontstyle 1
        fontsize 16
        font arial "Arial"
        drawString yLabel
        call resetFormat

    /** Draws the axes.
    */
    ::method drawAxles
        expose chartHeight chartWidth
        address jdor
        /* draw axles lines */
        stroke strokeThick3 3
        moveTo .JBCconstants~spacerEdge .JBCconstants~spacerEdge
        drawLine .JBCconstants~spacerEdge (.JBCconstants~spacerEdge + chartHeight)
        moveTo .JBCconstants~spacerEdge (.JBCconstants~spacerEdge + chartHeight)
        drawLine (.JBCconstants~spacerEdge + chartWidth) (.JBCconstants~spacerEdge + chartHeight)
        call resetFormat

    /** Draws arrows on the axes.
    */
    ::method drawArrows
        expose chartWidth chartHeight
        address jdor
        /* create arrows */
        shape arrow1 path
        pathMoveTo arrow1 0 10
        pathLineTo arrow1 5 0
        pathLineTo arrow1 10 10
        pathClose arrow1
        pathClone arrow1 arrow2
        /* transform arrows */
        transform t1 (.JBCconstants~spacerEdge - 5) (.JBCconstants~spacerEdge - 10)
        pathTransform arrow1 t1
        rotate 90
        transform tf90
        pathTransform arrow2 tf90
        transform t2 (.JBCconstants~spacerEdge + chartWidth + 10) (.JBCconstants~spacerEdge + chartHeight - 5)
        pathTransform arrow2 t2
        /* place arrows */
        fillShape arrow1
        fillShape arrow2
        call resetFormat

    /** Adds labels to the axes.
    */
    ::method addLabels
        expose chartWidth xLabel charWidth chartHeight height yLabel
        address jdor
        /* x-axis */
        moveTo (chartWidth / 2 + .JBCconstants~spacerEdge - xLabel~length * charWidth) (chartHeight + .JBCconstants~spacerEdge + .JBCconstants~spacerEdge / 1.5)
        fontstyle 1
        fontsize 16
        font arial "Arial"
        drawString xLabel
        /* y-axis */
        translate (.JBCconstants~spacerEdge / 4) (height / 2 + yLabel~length * charWidth)
        moveto 0 0
        rotate 270
        drawString yLabel
        call resetFormat

    /** Determines the appropriate value format based on the value.
    *  @param value the value to be checked
    *  @return the format code
    */
    ::method checkValue
        use arg value
        NUMERIC DIGITS 20
        select
            when value < 1000 then
                fc = self~unitNames[1]
            when value < 1000000 then
                fc = self~unitNames[2]
            when value < 1000000000 then
                fc = self~unitNames[3]
            when value < 1000000000000 then
                fc = self~unitNames[4]
            when value < 1000000000000000 then
                fc = self~unitNames[5]
        end
        return fc

    /** Formats the value based on the format code.
    *  @param value the value to be formatted
    *  @param fc the format code
    *  @return the formatted value
    */
    ::method formatValue
        use arg value, fc
        NUMERIC DIGITS 20
        select case fc
            when self~unitNames[2] then
                lineValue = format(value / 1000, , 2)
            when self~unitNames[3] then
                lineValue = format(value / 1000000, , 2)
            when self~unitNames[4] then
                lineValue = format(value / 1000000000, , 2)
            when self~unitNames[5] then
                lineValue = format(value / 1000000000000, , 2)
            otherwise
                lineValue = value
        end
        return lineValue

/** baseChart Class
*  This class defines the common attributes and methods for a chart.
*/
::class baseChart
    ::attribute dataSet
    ::attribute title
    ::attribute width
    ::attribute height
    ::attribute background
    ::attribute handler
    ::attribute title?
    ::attribute legend?
    ::attribute unitNames

    /** Initialize the baseChart object.
    *  @param dataSet an array of data to plot
    *  @param title the title of the chart
    *  @param width the width of the window
    *  @param height the height of the window
    *  @param background (optional) the background color of the chart (default: "white")
    *  @param title? (optional) boolean flag to indicate if the title should be displayed (default: .true)
    *  @param legend? (optional) boolean flag to indicate if the legend should be displayed (default: .true)
    *  @param unitNames (optional) an array with the names for big numbers (>999)
    */
    ::method init
        expose dataSet title width height background title? legend? unitNames
        use arg dataSet, title, width, height, background = "white", title? = .true, legend? = .true, unitNames = (.array~of("", "in thous.", "in mil.", "in bil.", "in tril."))

    /** Get the line format based on lineFormat parameter.
    *  @param lineFormat the format code for line style
    *  @return Line style configuration
    */
    ::method getLineFormat
        use arg lineFormat
        address jdor
        select case lineFormat
            when 1 then
                return 'dashed 1.5 0 0 1 "(5,3,0,1)" 0'
            when 2 then
                return 'pointed 3 0 0 1 "(1.5,1.5,0,1)" 0'
            when 3 then
                return 'mixed 1.5 0 0 1 "(2.5,3,5,3)" 0'
            otherwise
                return 'line 1'
        end

    /** Add a legend to the chart.
    *  @param combiNum a number to enable combining chats
    *  @param width the width of the window
    *  @param title the title of the window
    *  @param colorL the color of the legend
    *  @param chartType the type of the chart
    *  @param lineFormat (optional) the line format number (default: 0)
    */
    ::method addLegend
        use arg combiNum, width, title, colorL, chartType, lineFormat = 0
        address jdor
        if combiNum = 0 then
                yL = 30
            else
                yL = 30 * combiNum
        color colorL
        if chartType = "LINECHART" | chartType = "POINTCHART" then
            do
                stroke self~getLineFormat(lineFormat)
                moveTo (width - .JBCconstants~spacerEdge * 1.8) (yL + 5)
                drawLine (width - .JBCconstants~spacerEdge * 1.8 + 15) (yL + 5)
            end
        else
            do
                shape colorBox rect (width - .JBCconstants~spacerEdge * 1.8) yL 10 10
                fillShape colorbox
            end
        call resetFormat
        moveTo (width - .JBCconstants~spacerEdge * 1.8 + 20) (yL +10)
        drawString title
        drawShape colorBox

    /** Save the chart as a PNG image file.
    */
    ::method save
        use arg filename = .nil
        call BsfCommandHandler "add", "JDOR", self~handler
        address jdor
        if filename = .nil then
            fileName = .java.lang.System~getProperty("user.home") || .file~separator || self~title || ".png"
        saveImage fileName
        parse source opsys +1   -- get operating system
        if opsys="W" then cmd = quote(fileName)             -- windows
            else if opsys="D" then cmd = "open"     quote(fileName)  -- MacOS (Darwin)
                else                   cmd = "xdg-open" quote(fileName)  -- default to Linux
        --address system cmd
        say "Chart saved at" cmd

    /** Print the chart with default print setting.
    */
    ::method print
        call BsfCommandHandler "add", "JDOR", self~handler
        address jdor
        printScaleToPage .true
        printImage

    /** Quote a path in double-quotes.
    *  @param path the path to be quoted
    *  @return the quoted path
    */
    ::routine quote         -- enquote path in double-quotes (to cater for paths that may have blanks)
        return '"' || arg(1) || '"'

/** columnChart Class
* This class represents a column chart and extends the baseChart class.
*/
::class columnChart subclass baseChart
    ::attribute xlabel
    ::attribute ylabel
    ::attribute color
    ::attribute axisLabels?

    /** Initialize the columnChart object.
    *  @param dataSet an array of data to plot
    *  @param title the title of the chart
    *  @param xLabel the label for the x-axis
    *  @param yLabel the label for the y-axis
    *  @param width the width of the window
    *  @param height the height of the window
    *  @param color (optional) the color of the columns (default: "lightblue")
    *  @param axisLabels? (optional) boolean flag to indicate if the axis labels should be displayed (default: .true)
    */
    ::method init
        expose xLabel yLabel color axisLabels?
        use arg dataSet, title, xLabel, yLabel, width, height, color = "lightblue", axisLabels? = .true
        self~init:super(dataSet, title, width, height)   -- let superclass do its work

    /** Draw the column chart.
    *  @param combiNum (optional) a number to enable combining chats (default: 0)
    *  @param charts (optional) an array of charts to combine (default: .nil)
    *  @param maxDataSet (optional) the maximum dataset of charts that will be combined (default: .nil)
    */
    ::method draw
        use arg combiNum = 0, charts = .nil, maxDataSet = .nil
        if charts = .nil then
            charts = .array~new(1)
        if maxDataSet = .nil then
            maxDataSet = self~dataSet
        /* create canvas & coordinate system */
        if combiNum <= 1 then
            do
                canvas = .canvas~new(self~title, self~width, self~height, self~background)
                self~handler = canvas~handler
                csB = .coorSys~new(self~class~id, maxDataSet, self~title, self~xLabel, self~yLabel, self~width, self~height, self~unitNames)
                if self~title? & combiNum = 0 then
                    csB~addTitle
                csB~drawYAxis
                csB~drawAxles
                if self~axisLabels? then
                    csB~addLabels
            end
        /* max yValue of all bar charts */
        maxYValue = self~dataSet~maxValue("y")
        do chart over charts
            maxYValue = max(maxYValue, chart~dataSet~maxValue("y"))
        end
        /* setup parameters */
        spacer = self~width * 0.02
        barNum = self~dataSet~size
        chartWidth = sizeXToChart(self~width) - spacer * (barNum + 1)
        chartHeight = sizeYToChart(self~height)
        barWidth = chartWidth / (barNum * charts~size)
        maxBarHeight = chartHeight
        /* draw bars */
        x = .JBCconstants~spacerEdge + spacer
        do data over self~dataSet~values
            address jdor
            /* calc positions */
            y = .JBCconstants~spacerEdge
            if combiNum > 1 then
                x += barWidth * (combiNum - 1)
            barHeight = (data~yValue / maxYValue) * maxBarHeight
            y += maxBarHeight - barHeight
            /* create bar */
            shape bar rect x y barWidth barHeight
            color self~color
            fillShape bar
            color black
            drawShape bar
            /* add labels */
            if combiNum <= 1 then
                do
                    moveTo (x + (barWidth * charts~size) / 2 - data~xValue~length * .JBCconstants~charLength) (y + barHeight + 20)
                    drawString data~xValue
                end
            /* add legend */
            if self~legend? then
                self~addLegend(combiNum, self~width, self~title, self~color, self~class~id)
            /* end the loop*/
            if combiNum <> 0 then
                x += barWidth * (charts~size - (combiNum - 1)) + spacer
            else
                x += barWidth + spacer
        end
        say "ColumnChart got drawn"

/** barChart Class
* This class represents a bar chart and extends the baseChart class.
*/
::class barChart subclass baseChart
    ::attribute xlabel
    ::attribute ylabel
    ::attribute color
    ::attribute axisLabels?

    /** Initialize the barChart object.
    *  @param dataSet an array of data to plot
    *  @param title the title of the chart
    *  @param xLabel the label for the x-axis
    *  @param yLabel the label for the y-axis
    *  @param width the width of the window
    *  @param height the height of the window
    *  @param color (optional) the color of the bars (default: "lightblue")
    *  @param axisLabels? (optional) boolean flag to indicate if the axis labels should be displayed (default: .true)
    */
    ::method init
        expose xLabel yLabel color axisLabels?
        use arg dataSet, title, xLabel, yLabel, width, height, color = "lightblue", axisLabels? = .true
        self~init:super(dataSet, title, width, height)   -- let superclass do its work

    /** Draw the bar chart.
    *  @param combiNum (optional) a number to enable combining chats (default: 0)
    *  @param charts (optional) an array of charts to combine (default: .nil)
    *  @param maxDataSet (optional) the maximum dataset of charts that will be combined (default: .nil)
    */
    ::method draw
        use arg combiNum = 0, charts = .nil, maxDataSet = .nil
        if charts = .nil then
            charts = .array~new(1)
        if maxDataSet = .nil then
            maxDataSet = self~dataSet
        /* create canvas & coordinate system */
        if combiNum <= 1 then
            do
                canvas = .canvas~new(self~title, self~width, self~height, self~background)
                self~handler = canvas~handler
                csHB = .coorSys~new(self~class~id, maxDataSet, self~title, self~xLabel, self~yLabel, self~width, self~height, self~unitNames)
                if self~title? & combiNum = 0 then
                    csHB~addTitle
                csHB~drawXAxis("y")
                csHB~drawAxles
                if self~axisLabels? then
                    csHB~addLabels
            end
        /* max yValue of all bar charts */
        maxYValue = self~dataSet~maxValue("y")
        do chart over charts
            maxYValue = max(maxYValue, chart~dataSet~maxValue("y"))
        end
        /* setup parameters */
        spacer = self~height * 0.02
        barNum = self~dataSet~size
        chartWidth = sizexToChart(self~width)
        chartHeight = sizeYToChart(self~height) - spacer * (barNum + 1)
        barHeight = chartHeight / (barNum * charts~size)
        maxBarWidth = chartWidth
        x = .JBCconstants~spacerEdge
        y = self~height - .JBCconstants~spacerEdge - barHeight - spacer
        do data over self~dataSet~values
            address jdor
            /* calc positions */
            if combiNum > 1 then
                y -= barHeight * (combiNum - 1)
            barWidth = (data~yValue / maxYValue) * maxBarWidth
            /* create bar */
            shape bar rect x y barWidth barHeight
            color self~color
            fillShape bar
            color black
            drawShape bar
            /* add labels */
            if combiNum <= 1 then
                do
                    translate x y
                    moveTo (-barHeight / 2 - data~xValue~length * .JBCconstants~charLength) (-10)
                    rotate 270
                    drawString data~xValue
                    transform reset
                end
            /* add legend */
            if self~legend? then
                self~addLegend(combiNum, self~width, self~title, self~color, self~class~id)
            /* end the loop*/
            if combiNum <> 0 then
                y -= barHeight * (charts~size - (combiNum - 1)) + spacer
            else
                y -= barHeight + spacer
        end
        say "BarChart got drawn"

/** lineChart Class
* This class represents a line chart and extends the baseChart class.
*/
::class lineChart subclass baseChart
    ::attribute xlabel
    ::attribute ylabel
    ::attribute color
    ::attribute axisLabels?
    ::attribute lineFormat
    ::attribute area?

    /** Initialize the lineChart object.
    *  @param dataSet an array of data to plot
    *  @param title the title of the chart
    *  @param xLabel the label for the x-axis
    *  @param yLabel the label for the y-axis
    *  @param width the width of the window
    *  @param height the height of the window
    *  @param color (optional) the color of the bars (default: "black")
    *  @param axisLabels? (optional) boolean flag to indicate if the axis labels should be displayed (default: .true)
    *  @param lineFormat (optional) the format the line has (default: 0)
    *  @param area? (optional) boolean flag to indicate if areas below the lines should be displayer (default: .true)
    */
    ::method init
        expose xLabel yLabel color axisLabels? lineFormat area?
        use arg dataSet, title, xLabel, yLabel, width, height, color = "black", axisLabels? = .true, lineFormat = 0, area? = .true
        self~init:super(dataSet, title, width, height)   -- let superclass do its work

    /** Draw the line chart.
    *  @param combiNum (optional) a number to enable combining chats (default: 0)
    *  @param charts (optional) an array of charts to combine (default: .nil)
    *  @param maxDataSet (optional) the maximum dataset of charts that will be combined (default: .nil)
    */
    ::method draw
        use arg combiNum = 0, charts = .nil, maxDataSet = .nil
        if charts = .nil then
            charts = .array~new(1)
        if maxDataSet = .nil then
            maxDataSet = self~dataSet
        /* create canvas & coordinate system */
        csL = .coorSys~new(self~class~id, maxDataSet, self~title, self~xLabel, self~yLabel, self~width, self~height, self~unitNames)
        if combiNum <= 1 then
            do
                canvas = .canvas~new(self~title, self~width, self~height, self~background)
                self~handler = canvas~handler
                if self~title? & combiNum = 0 then
                    csL~addTitle
                csL~drawYAxis
                csL~drawAxles
                if self~axisLabels? then
                    csL~addLabels
            end
        if combiNum > 0  & charts~size = 1 then
            do
                csL~dataSet = self~dataSet
                csL~drawYAxis2(self~axisLabels?)
            end
        if combiNum > 2 & charts~size > 1 then
            csL~drawYAxis2(self~axisLabels?)
        /* max yValue of all point charts */
        maxYValue = self~dataSet~maxValue("y")
        do chart over charts
            maxYValue = max(maxYValue, chart~dataSet~maxValue("y"))
        end
        /* setup parameters */
        spacer = self~width * 0.02
        pointNum = self~dataSet~size
        chartWidth = sizeXToChart(self~width) - spacer * (pointNum + 1)
        chartHeight = sizeYToChart(self~height)
        barWidth = chartWidth / pointNum
        maxPointHeight = chartHeight
        prevX = .nil
        /* draw points */
        x = .JBCconstants~spacerEdge + spacer + barWidth / 2
        do data over self~dataSet~values
            address jdor
            /* calc positions */
            y = .JBCconstants~spacerEdge
            pointHeight = (data~yValue / maxYValue) * maxPointHeight
            y += maxPointHeight - pointHeight
            /* create point */
            shape point elli (x - 4) (y - 4) 8 8
            color self~color
            fillShape point
            drawShape point
            /* lines */
            moveTo x y
                /* special format */
            format = self~getLineFormat(self~lineFormat)
            stroke format
                /* draw line */
            if prevX <> .nil then
                do
                    drawLine prevX prevY
                    if self~area? then
                        do
                            /* create areas below line */
                                /* get rgb values */
                            color
                            rcColor = rc~toString
                            parse var rcColor . "=" r "," . "=" g "," . "=" b "]"
                                /* create color for area */
                            color areaColor r g b 100
                            color areaColor
                                /* create areas */
                            shape area path
                            pathMoveTo area x (y + 1)
                            pathLineto area prevX (prevY + 1)
                            pathLineto area prevX (.JBCconstants~spacerEdge + chartHeight)
                            pathLineto area x (.JBCconstants~spacerEdge + chartHeight)
                            pathClose area
                            fillShape area
                        end
                end
            prevX = x
            prevY = y
            call resetFormat
            /* add labels */
            if combiNum <= 1 then
                do
                    moveTo (x - data~xValue~length * .JBCconstants~charLength) (y + pointHeight + 20)
                    drawString data~xValue
                end
            /* add legend */
            if self~legend? then
                self~addLegend(combiNum, self~width, self~title, self~color, self~class~id, self~lineFormat)
            /* end the loop*/
            x += barWidth + spacer
        end
        say "LineChart got drawn"

/** pointChart Class
* This class represents a point chart and extends the baseChart class.
*/
::class pointChart subclass baseChart
    ::attribute xlabel
    ::attribute ylabel
    ::attribute color
    ::attribute lines?
    ::attribute points?
    ::attribute axisLabels?
    ::attribute lineFormat

    /** Initialize the pointChart object.
    *  @param dataSet an array of data to plot
    *  @param title the title of the chart
    *  @param xLabel the label for the x-axis
    *  @param yLabel the label for the y-axis
    *  @param width the width of the window
    *  @param height the height of the window
    *  @param color (optional) the color of the bars (default: "black")
    *  @param lines? (optional) boolean flag to indicate if lines should be displayed (default: .true)
    *  @param points? (optional) boolean flag to indicate if points should be displayed (default: .true)
    *  @param axisLabels? (optional) boolean flag to indicate if the axis labels should be displayed (default: .true)
    *  @param lineFormat (optional) the format the line has (default: 0)
    */
    ::method init
        expose xLabel yLabel color lines? points? axisLabels? lineFormat
        use arg dataSet, title, xLabel, yLabel, width, height, color = "black", lines? = .true, points? = .true, axisLabels? = .true, lineFormat = 0
        self~init:super(dataSet, title, width, height)   -- let superclass do its work

    /** Draw the point chart.
    *  @param combiNum (optional) a number to enable combining chats (default: 0)
    *  @param charts (optional) an array of charts to combine (default: .nil)
    *  @param maxDataSet (optional) the maximum dataset of charts that will be combined (default: .nil)
    */
    ::method draw
        use arg combiNum = 0, charts = .nil, maxDataSet = .nil
        if charts = .nil then
            charts = .array~new(1)
        if maxDataSet = .nil then
            maxDataSet = self~dataSet
        /* create canvas & coordinate system */
        if combiNum <= 1 then
            do
                canvas = .canvas~new(self~title, self~width, self~height, self~background)
                self~handler = canvas~handler
                csL = .coorSys~new(self~class~id, maxDataSet, self~title, self~xLabel, self~yLabel, self~width, self~height, self~unitNames)
                if self~title? & combiNum = 0 then
                    csL~addTitle
                csL~drawXAxis
                csL~drawYAxis
                csL~drawAxles
                csL~drawArrows
                if self~axisLabels? then
                    csL~addLabels
            end
        if combiNum = 2 & charts~size = 1 then
            do
                csL = .coorSys~new(self~class~id, self~dataSet, self~title, self~xLabel, self~yLabel, self~width, self~height, self~unitNames)
                csl~drawYAxis2(self~axisLabels?)
            end
        /* max xValue of all bar charts */
        maxXValue = self~dataSet~maxValue("x")
        do chart over charts
            maxXValue = max(maxXValue, chart~dataSet~maxValue("x"))
        end
        /* max yValue of all bar charts */
        maxYValue = self~dataSet~maxValue("y")
        do chart over charts
            maxYValue = max(maxYValue, chart~dataSet~maxValue("y"))
        end
        /* draw lines & points */
        address jdor
        prevData = .nil
        do data over self~dataSet~values
            /* points */
            color self~color
            moveTo calcXPos(data~xValue, maxXValue, self~width) calcYPos(data~yValue, maxYValue, self~height)
            if self~points? then
                do
                    shape point elli (calcXPos(data~xValue, maxXValue, self~width) - 4) (calcYPos(data~yValue, maxYValue, self~height) - 4) 8 8
                    fillShape point
                end
            /* lines */
            if self~lines? then
                do
                    /* special format */
                    format = self~getLineFormat(self~lineFormat)
                    stroke format
                    /* draw line */
                    if prevData <> .nil then
                        drawLine calcXPos(prevData~xValue, maxXValue, self~width) calcYPos(prevData~yValue, maxYValue, self~height)
                    prevData = data
                end
            call resetFormat
            /* add legend */
            if self~legend? then
                self~addLegend(combiNum, self~width, self~title, self~color, self~class~id, self~lineFormat)
        end
        say "PointChart got drawn"

/** pieChart Class
* This class represents a pie chart and extends the baseChart class.
*/
::class pieChart subclass baseChart
    ::attribute colorPallet

    /** Draw the pie chart.
    */
    ::method draw
        /* create canvas & coordinate system */
        canvas = .canvas~new(self~title, self~width, self~height, self~background)
        self~handler = canvas~handler
        cs = .coorSys~new(self~class~id, self~dataSet, self~title, , , self~width, self~height, self~unitNames)
        if self~title? then
            cs~addTitle
        address jdor
        d = self~height / 1.3
        /* draw diagram */
        percentSum = 0
        loop = 1
        index = 1
        do data over self~dataSet~values

            /* calc angles for segments*/
            percent = data~yValue / self~dataSet~sum
            startAngle = 360 - percentSum
            sweepAngle = -(360 * percent)

            /* add legend */
            if self~legend? then
                do
                    shape colorBox rect (self~width * 0.76) (30 * loop) 10 10
                    moveTo (self~width * 0.76 + 20) (30 * loop + 10)
                    drawString pp(data~xValue) format(percent * 100, , 2) "%"
                end

            /* create segments */
            shape segment arc (-self~height / 2.6) (-self~height / 2.6) d d startAngle sweepAngle pie
            percentSum += 360 * percent
            loop += 1

            /* auto select color */
            color myColor (random(1, 100)/100) (random(1, 100)/100) (random(1, 100)/100) (random(1, 100)/100)
            /* use own colorpallet */
            if self~colorPallet <> "COLORPALLET" then
                do
                    color self~colorPallet[index]
                    if index = self~colorPallet~size then
                        index = 1
                    else
                        index += 1
                end

            /* fill/draw segments + legend */
            translate (self~width / 2) (self~height / 2)
            fillShape segment
            transform reset
            fillshape colorBox
            color black
            translate (self~width / 2) (self~height / 2)
            drawShape segment
            transform reset
            drawShape colorBox

        end
        if self~class~id = "PIECHART" then
            say "PieChart got drawn"
        else
            say "RingChart got drawn"

/** ringChart Class
* This class represents a ring chart and extends the pieChart class.
*/
::class ringChart subclass pieChart
    ::attribute unit

    /** Initialize the ringChart object.
    *  @param dataSet an array of data to plot
    *  @param title the title of the chart
    *  @param width the width of the window
    *  @param height the height of the window
    *  @param unit (optional) display the unit of the data (default: "units")
    */
    ::method init
        expose unit
        use arg dataSet, title, width, height, unit = "units"
        self~init:super(dataSet, title, width, height)   -- let superclass do its work

    /** Draw the ring chart.
    */
    ::method draw
        self~draw:super   -- let superclass do its work
        address jdor
        /* fill middle with background */
        translate (self~width / 2) (self~height / 2)
        shape circle elli (-self~height / 4.6) (-self~height / 4.6) (self~height / 2.3) (self~height / 2.3)
        color white
        fillShape circle
        color black
        drawShape circle
        call resetFormat
        /* display total */
        display = "Total:" self~dataSet~sum self~unit
        translate (self~width / 2) (self~height / 2)
        moveTo (0 - display~length * .JBCconstants~charLength) 0
            /* format */
        fontstyle 1
        fontsize 16
        font arial "Arial"
        drawString display
        call resetFormat

/** multiChart Class
* This class represents a multi chart, made of multible charts of the same type.
*/
::class multiChart
    ::attribute title
    ::attribute charts

    /** Initialize the multiChart object by adding given charts to an array.
    *  @param chart1 the first chart
    *  @param chart2 the second chart
    *  @param chart3 the third chart
    *  @param chart4 the fourth chart
    *  @param chart5 the fifth chart
    *  @param title (optional) the title of the chart
    */
    ::method init
        expose charts title chart1
        use arg chart1, chart2, chart3, chart4, chart5, title = "Untitled"
        /* chart-dataSet */
        charts = .array~new
        if chart1 \= .nil then charts~append(chart1)
        if chart2 \= .nil then charts~append(chart2)
        if chart3 \= .nil then charts~append(chart3)
        if chart4 \= .nil then charts~append(chart4)
        if chart5 \= .nil then charts~append(chart5)

    /** Draw the multi chart.
    *  @param combiNum a number to enable combining chats
    */
    ::method draw
        expose charts chart1
        use arg combiNum = 0
        if combiNum = 1 then
            self~title = ""
        if combiNum > 0 then --adjust adding 1 in mixedChart
            combiNum -= 1
        /* max yValue of all bar charts */
        maxYValue = 0
        maxChart = chart1
        do chart over charts
            maxYValueO = maxYValue
            maxYValue = max(maxYValue, chart~dataSet~maxValue("y"))
            if maxYValueO < maxYValue then
                maxChart = chart
        end
        /* draw charts */
        do counter i chart over charts
            chart~draw(i + combiNum, charts, maxChart~dataSet)
        end
        if combiNum = 0 then
            do
                csMu = .coorSys~new(chart1~class~id, chart1~dataSet, self~title, chart1~xLabel, chart1~yLabel, chart1~width, chart1~height, chart1~unitNames)
                csMu~addTitle
            end
        say "MultiChart with" charts~size "["chart1~class~id"s] is finished"

/** stackChart Class
* This class stacks multible charts on top of each other.
*/
::class stackChart
    ::attribute title
    ::attribute toPercent

    /** Initialize the stackChart object by adding given charts to an array.
    *  @param chart1 the first chart
    *  @param chart2 the second chart
    *  @param chart3 the third chart
    *  @param chart4 the fourth chart
    *  @param chart5 the fifth chart
    *  @param title (optional) the title of the chart
    *  @param toPercent (optional) boolean flag to indicate if the chart should represent the data in 100 percent (default: .false)
    */
    ::method init
        expose charts title toPercent chart1
        use arg chart1, chart2, chart3, chart4, chart5, title = "Untitled", toPercent = .false
        /* chart-dataSet */
        charts = .array~new
        if chart1 \= .nil then charts~append(chart1)
        if chart2 \= .nil then charts~append(chart2)
        if chart3 \= .nil then charts~append(chart3)
        if chart4 \= .nil then charts~append(chart4)
        if chart5 \= .nil then charts~append(chart5)

    /** Draw stacked chart by calling corresponding method.
    */
    ::method draw
        expose chart1
        select case chart1~class~id
            when "COLUMNCHART" then
                self~drawColumn
            when "BARCHART" then
                self~drawBar
            when "LINECHART" then
                self~drawLine
            otherwise
                say "You can only stack [ColumnChart], [BarChart] and [LineChart]"
        end

    /** Draw the stacked column chart.
    */
    ::method drawColumn
        expose charts chart1
        /* create array for each chart with every yValue */
        chartsData = .array~new
        do chart over charts
            datas = .array~new
            do counter i data over chart~dataSet~values
                datas[i] = data~yValue
            end
            chartsData~append(datas)
        end
        /* sumData stores the sum of each equivalent yValues */
        sumData = .array~new
            /* fill sumData with enough data to calc in next step */
        do fill = 1 to datas~size
            sumData~append(0)
        end
        /* sum-up the equivalent yValues */
        do i = 1 to chartsData~size
            do m = 1 to datas~size
                sumData[m] += chartsData[i][m]
            end
        end
        /* find max Value */
        maxValue = 0
        do y over sumData
            maxValue = max(maxValue, y)
        end

        /* package maxValue for sending it to coorSys */
        maxDataSet = .dataSet~new
        if self~toPercent then
            maxDataSet~addValue(0, 100) --show percentage
        else
            maxDataSet~addValue(0, maxValue)
        /* create window */
        .canvas~new(chart1~title, chart1~width, chart1~height, chart1~background)
        csSC = .coorSys~new(chart1~class~id, maxDataSet, self~title, chart1~xLabel, chart1~yLabel, chart1~width, chart1~height, chart1~unitNames)
        if chart1~title? then
            csSC~addTitle
        csSC~drawYAxis(self~toPercent)
        csSC~drawAxles
        if chart1~axisLabels? then
            csSC~addLabels
        /* setup parameters */
        spacer = chart1~width * 0.02
        barNum = chart1~dataSet~size
        chartWidth = sizeXToChart(chart1~width) - spacer * (barNum + 1)
        chartHeight = sizeYToChart(chart1~height)
        barWidth = chartWidth / barNum
        maxBarHeight = chartHeight
        /* draw charts */
        address jdor
        x = .JBCconstants~spacerEdge + spacer
        do c = 1 to chartsData[1]~size
            barHeightSum = 0
            do i = 1 to chartsData~size
                /* calc positions */
                if self~toPercent then
                    barHeight = (chartsData[i][c] / sumData[c]) * maxBarHeight
                else
                    barHeight = (chartsData[i][c] / maxValue) * maxBarHeight
                y = .JBCconstants~spacerEdge + maxBarHeight - barHeight - barHeightSum
                /* create bar */
                shape bar rect x y barWidth barHeight
                color charts[i]~color
                fillShape bar
                color black
                drawShape bar
                barHeightSum += barHeight
                /* add legend */
                if charts[i]~legend? then
                    charts[i]~addLegend(i, charts[i]~width, charts[i]~title, charts[i]~color, charts[i]~class~id)
            end
            /* add labels */
            moveTo (x + barWidth / 2 - charts[1]~dataSet~values[c]~xValue~length * .JBCconstants~charLength) (chartHeight + .JBCconstants~spacerEdge + 20)
            drawString charts[1]~dataSet~values[c]~xValue
            /* end loop */
            x += barWidth + spacer
            y = 0
        end
        say "StackChart with" charts~size "["chart1~class~id"s] is finished"

    /** Draw the stacked bar chart.
    */
    ::method drawBar
        expose charts chart1
        /* create array for each chart with every yValue */
        chartsData = .array~new
        do chart over charts
            datas = .array~new
            do counter i data over chart~dataSet~values
                datas[i] = data~yValue
            end
            chartsData~append(datas)
        end
        /* sumData stores the sum of each equivalent yValues */
        sumData = .array~new
            /* fill sumData with enough data to calc in next step */
        do fill = 1 to datas~size
            sumData~append(0)
        end
        /* sum-up the equivalent yValues */
        do i = 1 to chartsData~size
            do m = 1 to datas~size
                sumData[m] += chartsData[i][m]
            end
        end
        /* find max Value */
        maxValue = 0
        do y over sumData
            maxValue = max(maxValue, y)
        end

        /* package maxValue for sending it to coorSys */
        maxDataSet = .dataSet~new
        if self~toPercent then
            maxDataSet~addValue(0, 100)  --show percentage
        else
            maxDataSet~addValue(0, maxValue)
        /* create window */
        .canvas~new(chart1~title, chart1~width, chart1~height, chart1~background)
        csSC = .coorSys~new(chart1~class~id, maxDataSet, self~title, chart1~xLabel, chart1~yLabel, chart1~width, chart1~height, chart1~unitNames)
        if chart1~title? then
            csSC~addTitle
        csSC~drawXAxis("y", self~toPercent)
        csSC~drawAxles
        if chart1~axisLabels? then
            csSC~addLabels
        /* create array to save temp data */
        prevWidth = .array~new
        do i = 1 to sumData~size
            prevWidth~append(0)
        end
        /* setup parameters */
        spacer = chart1~height * 0.02
        barNum = chart1~dataSet~size
        chartWidth = sizexToChart(chart1~width)
        chartHeight = sizeYToChart(chart1~height) - spacer * (barNum + 1)
        barHeight = chartHeight / barNum
        maxBarWidth = chartWidth
        do counter c chart over charts
            x = .JBCconstants~spacerEdge
            y = chart1~height - .JBCconstants~spacerEdge - barHeight - spacer
            do counter i data over chart~dataSet~values
                address jdor
                /* calc positions */
                if self~toPercent then
                    barWidth = (data~yValue / sumData[i]) * maxBarWidth
                else
                    barWidth = (data~yValue / maxValue) * maxBarWidth
                /* create bar */
                shape bar rect (x + prevWidth[i]) y barWidth barHeight
                color chart~color
                fillShape bar
                color black
                drawShape bar
                /* add labels */
                if c = 1 then
                    do
                        translate x y
                        moveTo (-barHeight / 2 - data~xValue~length * .JBCconstants~charLength) (-10)
                        rotate 270
                        drawString data~xValue
                        transform reset
                    end
                /* add legend */
                if chart~legend? then
                    chart~addLegend(c, chart~width, chart~title, chart~color, chart~class~id)
                /* end the loop*/
                y -= barHeight + spacer
                prevWidth[i] += barWidth
            end
        end
        say "StackChart with" charts~size "["chart1~class~id"s] is finished"

    /** Draw the stacked line chart.
    */
    ::method drawLine
        expose charts chart1
        /* create array for each chart with every yValue */
        chartsData = .array~new
        do chart over charts
            datas = .array~new
            do counter i data over chart~dataSet~values
                datas[i] = data~yValue
            end
            chartsData~append(datas)
        end
        /* sumData stores the sum of each equivalent yValues */
        sumData = .array~new
            /* fill sumData with enough data to calc in next step */
        do fill = 1 to datas~size
            sumData~append(0)
        end
        /* sum-up the equivalent yValues */
        do i = 1 to chartsData~size
            do m = 1 to datas~size
                sumData[m] += chartsData[i][m]
            end
        end
        /* find max Value */
        maxValue = 0
        do y over sumData
            maxValue = max(maxValue, y)
        end
        /* package maxValue for sending it to coorSys */
        maxDataSet = .dataSet~new
        if self~toPercent then
            maxDataSet~addValue(0, 100)  --show percentage
        else
            maxDataSet~addValue(0, maxValue)
        /* create window */
        .canvas~new(chart1~title, chart1~width, chart1~height, chart1~background)
        csSC = .coorSys~new(chart1~class~id, maxDataSet, self~title, chart1~xLabel, chart1~yLabel, chart1~width, chart1~height, chart1~unitNames)
        if chart1~title? then
            csSC~addTitle
        csSC~drawYAxis(self~toPercent)
        csSC~drawAxles
        if chart1~axisLabels? then
            csSC~addLabels
        /* create arrays for temp values */
        prevHeight = .array~new
        pPrevHeight = .array~new
        do c = 1 to sumData~size
            prevHeight~append(0)
            pPrevHeight~append(0)
        end
        /* drawing the chart */
        do counter c chart over charts
            /* setup parameters */
            spacer = chart1~width * 0.02
            pointNum = chart1~dataSet~size
            chartWidth = sizeXToChart(chart1~width) - spacer * (pointNum + 1)
            chartHeight = sizeYToChart(chart1~height)
            barWidth = chartWidth / pointNum
            maxPointHeight = chartHeight
            prevX = .nil
            prevY = .nil

            x = .JBCconstants~spacerEdge + spacer + barWidth / 2
            address jdor
            do counter i data over chart~dataSet~values
                /* calc position */
                y = .JBCconstants~spacerEdge
                if self~toPercent then
                    pointHeight = (data~yValue / sumData[i]) * maxPointHeight + prevHeight[i]
                else
                    pointHeight = (data~yValue / maxValue) * maxPointHeight + prevHeight[i]
                y += maxPointHeight - pointHeight
                /* create point */
                shape point elli (x - 4) (y - 4) 8 8
                color chart~color
                fillShape point
                drawShape point
                /* lines */
                moveTo x y
                    /* special format */
                format = chart~getLineFormat(chart~lineFormat)
                stroke format
                    /* draw line */
                if prevX <> .nil then
                    do
                        drawLine prevX prevY
                        if chart~area? then
                            do
                                /* create areas below line */
                                    /* get rgb values */
                                color
                                rcColor = rc~toString
                                parse var rcColor . "=" r "," . "=" g "," . "=" b "]"
                                    /* create color for area */
                                color areaColor r g b 100
                                color areaColor
                                    /* create areas */
                                shape area path
                                pathMoveTo area x (y + 1)
                                pathLineto area prevX (prevY + 1)
                                if c = 1 then
                                    pathLineto area prevX (.JBCconstants~spacerEdge + chartHeight)
                                else
                                    pathLineto area prevX (prevY + prevHeight[i-1] - pPrevHeight[i-1])

                                if c = 1 then
                                    pathLineto area x (.JBCconstants~spacerEdge + chartHeight)
                                else
                                    pathLineto area x (y + (pointHeight - prevHeight[i]))
                                pathClose area
                                fillShape area
                            end
                    end
                /* end the loop */
                prevX = x
                prevY = y
                pPrevHeight[i] = prevHeight[i]
                prevHeight[i] = pointHeight
                call resetFormat
                /* add labels */
                moveTo (x - data~xValue~length * .JBCconstants~charLength) (y + pointHeight + 20)
                drawString data~xValue
                /* add legend */
                if chart~legend? then
                    chart~addLegend(c, chart~width, chart~title, chart~color, chart~class~id, chart~lineFormat)
                /* end the loop*/
                x += barWidth + spacer
            end
        end
        say "StackChart with" charts~size "["chart1~class~id"s] is finished"

/** This class represent a chart made of a line chart or multichart of line charts
* and column chart or multichart of column charts.
*/
::class mixedChart
    ::attribute title

    /** Constructor, fetches chart1 and chart2.
    *  @param chart1 the first chart
    *  @param chart2 the second chart
    */
    ::method init
        expose chart1 chart2
        use arg chart1, chart2, title = "Untitled"
        /* correct wrong chartType order of user */
        if chart2~class~id = "COLUMNCHART" then
            do
                chart0 = chart1
                chart1 = chart2
                chart2 = chart0
            end

    /** Draw the mixed chart.
    */
    ::method draw
        expose chart1 chart2
        i = 2
        if chart1~class~id = "MULTICHART" then --if multiChart legend has to be adjusted
            i = chart1~charts~size + 1
        chart1~draw(1)
        chart2~draw(i)
        if chart1~class~id = "MULTICHART" then
            csM = .coorSys~new(chart1~charts[1]~class~id, chart1~charts[1]~dataSet, self~title, chart1~charts[1]~xLabel, chart1~charts[1]~yLabel, chart1~charts[1]~width, chart1~charts[1]~height, chart1~charts[1]~unitNames)
        else
            csm = .coorSys~new(chart1~class~id, chart1~dataSet, self~title, chart1~xLabel, chart1~yLabel, chart1~width, chart1~height, chart1~unitNames)
        csM~addTitle
        say "MixedChart with" pp(chart1~class~id) "and" pp(chart2~class~id) "is finished"

::requires 'BSF.CLS'

/** This routine calculates the chart width based on the window width.
*  @param size the value that should be adjusted
*  @return the adjusted value
*/
::routine sizeXToChart
    use arg size
    return size - .JBCconstants~spacerEdge * 3.5

/** This routine calculates the chart height based on the window height.
*  @param size the value that should be adjusted
*  @return the adjusted value
*/
::routine sizeYToChart
    use arg size
    return size - .JBCconstants~spacerEdge * 2

/** This routine calculates the x position in a point chart for a given value.
*  @param value the value for which the position should be calculated
*  @param maxValue the max x value of the dataset
*  @param height the height of the window
*  @return the y position in a point chart
*/
::routine calcXPos
    use arg value, maxValue, width
    return sizeXToChart(width) / maxValue * value + .JBCconstants~spacerEdge

/** This routine calculates the y position in a point chart for a given value.
*  @param value the value for which the position should be calculated
*  @param maxValue the max y value of the dataset
*  @param height the height of the window
*  @return the y position in a point chart
*/
::routine calcYPos
    use arg value, maxValue, heigth
    return sizeYToChart(heigth) - sizeYToChart(heigth) / maxValue * value + .JBCconstants~spacerEdge

/** This routine resets the format of jdor to default.
* Resets transformation, font and color, among others.
*/
::routine resetFormat
    address jdor
    transform reset
    fontstyle 0
    fontsize 12
    font courier "Courier"
    stroke default 1
    color black

/** This class provides constants that the entire program can access.
*/
::class JBCconstants
    /** This constant determines the distance between the window border and the chart.
    */
    ::constant spacerEdge 80
    /** This constant determines the approximate width of a displayed character to correctly adjust the position of text.
    */
    ::constant charLength 3.5
    /** This constant provides information about the framework's version.
    */
    ::constant version "1.0"