#!/usr/bin/env rexx
/* author:     Rony G. Flatscher
   date:       2009-09-10, 2009-10-11
   changed:    2015-06-26, rgf:
               - adjust code to new BsfRexxProxy() behaviour: if Java supplies a BSFRexxProxy() object
                 BSF4ooRexx automatically unwraps it and submits it as an argument

   license:    ASF 2.0 (Apache) or CPL 1.0

   purpose:    controller of the multithreaded execution of Rexx threads
*/

use arg rpTestUnit, rpArray, nrThreads, javaThreadName

    tid=bsfGetTID()              -- get this TID
--    testUnit=BsfRexxProxy(rpTestUnit, "ooRexxObject") -- get ooRexx object that gets proxied
    testUnit=rpTestUnit

          -- save this thread's TID
    argArray=bsf.createJavaArrayOf("java.lang.Object", tid)
    -- rpArray~sendMessage("APPEND", argArray)
    rpArray~"APPEND"(argArray)

    rexxArray=.array~new         -- create a new rexxArray for this controller's threads

    counter=.counter~new         -- create a counter object

    do i=1 to nrThreads
       counter~increase                   -- increase the counter
          -- thread will decrease counter before returning
       .ThreadClz_using_RexxProxy_Array~new~work(testUnit, tid, i, counter, rpArray, rexxArray, javaThreadName)
    end

    maxTimeToWait=1+nrThreads*1.5/10
    call time "Reset"            -- reset timer
    loop while counter~value>0, time("elapsed")<maxTimeToWait  -- until no more threads, or over 10seconds
       call sysSleep .001              -- yield
    end

    testUnit~assertEquals(0, counter~value, "active thread counter in Java thread:" javaThreadName)

      -- this controller created 'nrThreads' threads, check it
    testUnit~assertEquals(nrThreads, rexxArray~items, "total number of threads:" javaThreadName)


::requires BSF.CLS


/* ==================================================================================== */
-- implement a counter that one can increase/decrease
::class counter
::method init           -- constructor
  expose value
  value=0               -- set attribute
  forward to (super)    -- let superclass initialize

::attribute value       -- allow interfacing with the attribute

::method increase       -- increase value by 1
  expose value
  value+=1
  return value

::method decrease       -- decrease value by 1
  expose value
  value-=1
  return value



/* ==================================================================================== */
-- implement a class with a threaded worker method
::class "ThreadClz_using_RexxProxy_Array"
::method work
  use arg testUnit, TID, nr, counter, rpArray, rexxArray, javaThreadName
  reply                 -- return & create a new thread

  thisTID=BsfGetTID()
  testUnit~assertTrue(BsfAttachToTID(TID), "attaching to main TID:" TID", thread #" nr", thisTID="thisTID "running in Java thread named:" javaThreadName)

   -- have the RexxProxy send the following message to its proxied Rexx object
  argArray=bsf.createJavaArrayOf("java.lang.Object", thisTID)
  rpArray~"APPEND"(argArray) -- save this TID
  rexxArray~append(thisTID)

  testUnit~assertTrue(BsfDetach(), "detaching from main TID:" TID", thread #" nr", thisTID="thisTID "running in Java thread named:" javaThreadName)
  counter~decrease
  return